//=============================================================================
// SoR_RecentObtainedItemHighlight_MZ.js
// SoR License (C) 2020 蒼竜, REQUIRED User Registration on Dragon Cave
// http://dragonflare.blue/dcave/license.php
// ----------------------------------------------------------------------------
// Latest version v1.04 (2021/03/13)
//=============================================================================
/*:ja
@plugindesc ＜直近取得アイテムハイライト＞ v1.04
@author 蒼竜
@target MZ
@url http://dragonflare.blue.jp/dcave/
@help ゲーム中に新規に入手したアイテムを記録し、
アイテム画面の表示時にそれらを強調表示します。

@param ---全般---
@param MaxRecentObtained_Stack
@desc 新規取得物として記録するアイテムの種類数上限 (default: 5)
@type number
@default 5

@param ---アイコン強調---
@param UseStrokeRect_ItemIcons
@desc 'true': 新規取得物のアイコンを枠で囲む (default: true)
@default true
@type boolean

@param Color_StrokeRect
@desc 新規取得物のアイコンを囲む枠の色，Window.pngの番号に対応 (default: 17)
@type number
@default 17

@param ---強調テキスト---
@param UseTexts_RecentObtained
@desc 'true': 任意テキストで新規取得物を装飾する (default: true)
@default true
@type boolean

@param Texts_RecentObtained
@desc 装飾テキストの内容 (default: NEW)
@type string
@default NEW

@param Texts_Alignment
@desc 装飾テキスト描画の基準位置
@type select
@option 左寄せ
@value 0
@option 右寄せ
@value 1
@default 0

@param Texts_PositionX
@desc 装飾するテキストの表示位置x座標 (default: 20)
@type number
@default 20

@param Texts_PositionY
@desc 装飾するテキストの表示位置y座標 (default: 10)
@type number
@default 10

@param FontSize_Texts
@desc 装飾するテキストのフォントサイズ (default: 14)
@type number
@default 14

@param Color_Texts
@desc 装飾するテキストの文字色，Window.pngの番号に対応 (default: 6)
@type number
@default 6

@command ForceObtainedClear
@text 取得記録クリア[直近取得アイテムハイライト]
@desc 取得アイテムの記録を直ちに全消去します
*/
/*:
@plugindesc <Highlighting New Items on Menu> v1.04
@author Soryu
@target MZ
@url http://dragonflare.blue.jp/dcave/index_e.php
@help This plugin observes the item obtain process during the game 
to record newly obtained items and emphasize them in the item window menus.

@param ---General---
@param MaxRecentObtained_Stack
@desc Number of items to record in the stack as recenly obtained. (default: 5)
@type number
@default 5

@param ---Highlight Icon---
@param UseStrokeRect_ItemIcons
@desc If 'true', the icon of recently obtained items is highlighted in the menu. (default: true)
@default true
@type boolean

@param Color_StrokeRect
@desc Color associated with color palette in "Window.png" for highlighting icon (default: 17)
@type number
@default 17

@param ---Highlight Text---
@param UseTexts_RecentObtained
@desc If 'true', arbitrary texts to highlight items are available. (default: true)
@default true
@type boolean

@param Texts_RecentObtained
@desc A text to put on the recenlty obtained item (default: NEW)
@type string
@default NEW

@param Texts_Alignment
@desc Criteria of highlight text position
@type select
@option Left aligned
@value 0
@option Right aligned
@value 1
@default 0

@param Texts_PositionX
@desc Position of highlighting text for X-coordinate (default: 20)
@type number
@default 20

@param Texts_PositionY
@desc Position of highlighting text for Y-coordinate (default: 10)
@type number
@default 10

@param FontSize_Texts
@desc Font size of highlighting text (default: 14)
@type number
@default 14

@param Color_Texts
@desc Color associated with color palette in "Window.png" for highlighting text (default: 6)
@type number
@default 6

@command ForceObtainedClear
@text Clear Record[Highlighting New Items on Menu]
@desc Clear the stack of recently obtained items immediately.
*/


(function() {
const pluginName = "SoR_RecentObtainedItemHighlight_MZ";
const Param = PluginManager.parameters(pluginName);

const MaxRecentObtained_Stack = Number(Param['MaxRecentObtained_Stack']) || 5; 
const UseStrokeRect_ItemIcons = Boolean(Param['UseStrokeRect_ItemIcons'] === 'true') || false;
const Color_StrokeRect = Number(Param['Color_StrokeRect']) || 0; 

const UseTexts_RecentObtained = Boolean(Param['UseTexts_RecentObtained'] === 'true') || false;
const Texts_RecentObtained = String(Param['Texts_RecentObtained']) || '';
const Texts_PositionX = Number(Param['Texts_PositionX']) || 0;
const Texts_PositionY = Number(Param['Texts_PositionY']) || 0;
const FontSize_Texts = Number(Param['FontSize_Texts']) || 0;
const Color_Texts = Number(Param['Color_Texts']) || 0;

//v1.01
const Texts_Alignment = Number(Param['Texts_Alignment']) || 0;


PluginManager.registerCommand(pluginName, "ForceObtainedClear", args => {
	$gameParty._recentObtained = [];
});

const SoR_ROI_GP_initAllItems = Game_Party.prototype.initAllItems;
Game_Party.prototype.initAllItems = function() {
    SoR_ROI_GP_initAllItems.call(this);
    this._recentObtained = [];
    this.IsActivateRecentObtained = true;
}


const SoR_ROI_GA_tradeItemWithParty = Game_Actor.prototype.tradeItemWithParty;
Game_Actor.prototype.tradeItemWithParty = function(newItem, oldItem) {
    $gameParty.IsActivateRecentObtained = false;
    const ret = SoR_ROI_GA_tradeItemWithParty.call(this,...arguments);
    $gameParty.IsActivateRecentObtained = true;
    return ret;
}

const SoR_ROI_GP_gainItem = Game_Party.prototype.gainItem;
Game_Party.prototype.gainItem = function(item, amount, includeEquip) {
    SoR_ROI_GP_gainItem.call(this, ...arguments);

    if(typeof this.IsActivateRecentObtained === "undefined") this.IsActivateRecentObtained=true;
    if(this.IsActivateRecentObtained) this.processRecentObtain(...arguments);
}

Game_Party.prototype.processRecentObtain = function(item, amount, includeEquip) {

    const container = this.itemContainer(item);
    if(amount>0 && container){
        if(!this._recentObtained) this._recentObtained = [];
        else{
            let idx=-1;
            const nro = this._recentObtained.length;
            const stritem = JSON.stringify(item);
            for(let i = 0; i < nro; i++){
                if(stritem == JSON.stringify(this._recentObtained[i])){
                    idx = i;
                    break;
                }
            }
            if(idx!=-1) this._recentObtained.splice(idx,1);
        }
        this._recentObtained.push(item);
        if(this._recentObtained.length > MaxRecentObtained_Stack) this._recentObtained.shift();
    }

}

const SoR_ROI_WIL_drawItem = Window_ItemList.prototype.drawItem;
Window_ItemList.prototype.drawItem = function(index) {
    SoR_ROI_WIL_drawItem.call(this, ...arguments);

    const item = this.itemAt(index);
    if(!item) return;

    if(!$gameParty._recentObtained) $gameParty._recentObtained = []; 
    const stritem = JSON.stringify(item);
    if($gameParty._recentObtained.some(x => JSON.stringify(x)== stritem)) this.drawRecentObtainedIndicator(item, index);
}

Window_ItemList.prototype.drawRecentObtainedIndicator = function(item, index) {
        const recti = this.itemRect(index); 
        const rect = this.itemLineRect(index);

        if(UseStrokeRect_ItemIcons) this.drawEmphasisRect(rect.x-2, rect.y,36,36);

        if(UseTexts_RecentObtained){
            this.contents.fontSize = FontSize_Texts;
            this.changeTextColor(ColorManager.textColor(Color_Texts));

            const align = Texts_Alignment==0? "left": "right";
            const tx = align=="left"? (rect.x+Texts_PositionX) : (rect.x+recti.width-Texts_PositionX-this.textWidth(Texts_RecentObtained));
            this.drawText(Texts_RecentObtained, tx, rect.y+Texts_PositionY, recti.width);
            this.resetTextColor();
            this.resetFontSettings();
        }
}

Window_ItemList.prototype.drawEmphasisRect = function(x, y, width, height) {
    const outlineColor = ColorManager.textColor(7);
    const mainColor = ColorManager.textColor(Color_StrokeRect);

    this.contents.strokeRect(x, y, width, height, outlineColor);
    this.contents.strokeRect(x + 1, y + 1, width - 2, height - 2, mainColor);
}

})();