//=============================================================================
// SoR_BattleComboCounter_MZ.js
// SoR License (C) 2020 蒼竜, REQUIRED User Registration on Dragon Cave
// http://dragonflare.blue/dcave/license.php
// ----------------------------------------------------------------------------
// Latest version v1.10 (2021/10/05)
//=============================================================================
/*:ja
@plugindesc ＜個別コンボカウンター＞ v1.10
@author 蒼竜
@target MZ
@url http://dragonflare.blue/dcave/
@orderAfter SoR_BattleTeminationRefactor_MZ
@orderBefore SoR_VictoryPerformance_MZ
@base SoR_BattleTeminationRefactor_MZ
@help ※要57.「戦闘終了処理ルーチン細分化」(SoR_BattleTeminationRefactor_MZ.js)

戦闘中、連続で発生した攻撃(ダメージ)判定を取得し
連続ヒット数として画面に表示します。また、連続ヒット数を
戦闘終了時に評価し、ボーナス経験値を発生させます。
連続ヒット数は、各エネミーごとにカウントされます。

※注意(プラグイン配置順番により一部演算が変わります)※
4.「バトラー種族定義」より上(最終ダメージにコンボ倍率反映)
同より下(コンボ補正後に種族倍率)

@param ---全般---
@param Combo_Y-padd
@desc コンボカウンター表示y座標補正(default: 0)
@default 0
@min -9999
@type number
@param Show_TotalDamage
@desc 'true': 合計ダメージ数を併記する
@default false
@type boolean
@param Total_Y-padd
@desc トータルダメージ表示y座標(本体とトータル部分の差分)補正
@default 0
@min -9999
@type number

@param Show_ComboBonusRate
@desc 'true': コンボダメージボーナス倍率を併記する
@default false
@type boolean
@param Bonus_Y-padd
@desc ダメージボーナス表示y座標(トータル部分とボーナス部分の差分)補正
@default 0
@min -9999
@type number

@param ---画像関係(必須)---
@param ImageFile_Base
@desc コンボ土台背景 (default: "cc1")
@type file
@dir img/SoRBatHUD/
@default cc1
@param ImageFile_Count
@desc コンボヒット数数値 (default: "cc2")
@type file
@dir img/SoRBatHUD/
@default cc2
@param ImageFile_Hits
@desc コンボヒット"HIT"画像 (default: "cc3")
@type file
@dir img/SoRBatHUD/
@default cc3
@param ImageFile_Total
@desc コンボ"トータル"画像 (default: "cc4")
@type file
@dir img/SoRBatHUD/
@default cc4
@param ImageFile_TotalCount
@desc コンボトータルダメージ数値 (default: "cc5")
@type file
@dir img/SoRBatHUD/
@default cc5

@param ImageFile_RateText
@desc コンボ"ダメージボーナス"画像 (default: "cc6")
@type file
@dir img/SoRBatHUD/
@default cc6
@param ImageFile_RateCount
@desc コンボダメージボーナス数値 (default: "cc7")
@type file
@dir img/SoRBatHUD/
@default cc7

@param -コンボ継続-
@param Combo_Duration
@desc コンボ(連続ヒット)判定継続時間 (default: 120)
@default 120
@type number
@param Combo_DurationTiming
@desc コンボ継続時間経過タイミング (default: 3)
@type select
@option 常に(フルアクティブ)
@value 0
@option コマンド入力中は止める
@value 1
@option アクション中は止める
@value 2
@option アクション・コマンド入力中は止める
@value 3
@default 3

@param -中断条件など-
@param StopCombo_MissAttack
@desc 'true': 攻撃ミスでコンボ中断
@default false
@type boolean
@param StopCombo_TargetAction
@desc 'true': 対象の行動開始でコンボ中断
@default false
@type boolean
@param SuspendCombo_TargetGuard
@desc 'true': 対象が防御中は命中してもコンボカウントしない
@default false
@type boolean

@param -コンボダメージボーナス-
@param ComboBonus_StartHit
@desc ボーナスダメージ発生開始ヒット数。1以下で無効 (default: 1)
@default 1
@type number
@param ComboBonus_DamageRate
@desc コンボによるダメージ倍率ボーナス値(%) y=an+100のaに相当 (default: 3)
@default 3
@type number

@param --指定ヒット達成--
@param Rewards_MaxHit
@desc 指定ヒット数達成時に特殊処理を発生させる。0で無効 (default: 0)
@default 0
@type number
@param Rewards_Process
@desc 指定ヒット数達成時に実行する特殊処理。スクリプトテキスト
@default 
@type string

@param -コンボ経験値ボーナス-
@param UseComboBounsEXP
@desc 'true': 連続ヒット数による経験値ボーナスを発生させる。
@default false
@type boolean
@param Formula_ComboBounsEXP
@desc ボーナス経験値式。ヒット数"n"による多項式を記述
@default n*(n+6)/6
@type string
*/
/*:
@plugindesc <Individual Combo Count System> v1.10
@author Soryu
@target MZ
@url http://dragonflare.blue/dcave/index_e.php
@orderAfter SoR_BattleTeminationRefactor_MZ
@orderBefore SoR_VictoryPerformance_MZ
@base SoR_BattleTeminationRefactor_MZ
@help [Prerequisite] 57. SoR_BattleTeminationRefactor_MZ

This plugin observes the attack hits for EACH BATTLER
during the battle scene, and count consecutive hits
within designated duration to show on the game screen.

At the end of battle, maximum chained hits are evaluated
to present EXP bonus and other benefits.

!!CAUTION (Some systems are affected by the plugin order)!!
Above 4. "SoR_RaceforBattler" 
-> Combo damage bonus is applied to the final damage.
Below 4. "SoR_RaceforBattler" 
-> Race effectivity is applied after Combo damage bonus application.

@param ---General---
@param Combo_Y-padd
@desc Padding of combo UI for y-coordinate (default: 0)
@default 0
@min -9999
@type number
@param Show_TotalDamage
@desc If true, total damage is also displayed.
@default false
@type boolean
@param Total_Y-padd
@desc Padding of total damage UI (between the combo base and total damage)
@default 0
@min -9999
@type number

@param Show_ComboBonusRate
@desc If true, combo damage rate is also displayed.
@default false
@type boolean
@param Bonus_Y-padd
@desc Padding of damage bonus UI (between the total damage and bonus)
@default 0
@min -9999
@type number


@param ---Combo Images---
@param ImageFile_Base
@desc Image source of Combo background (default: "cc1")
@type file
@dir img/SoRBatHUD/
@default cc1
@param ImageFile_Count
@desc Image source of Combo hit digits (default: "cc2")
@type file
@dir img/SoRBatHUD/
@default cc2
@param ImageFile_Hits
@desc Image source of Combo "HITS" text (default: "cc3")
@type file
@dir img/SoRBatHUD/
@default cc3
@param ImageFile_Total
@desc Image source of Combo "TotalDamage" text (default: "cc4")
@type file
@dir img/SoRBatHUD/
@default cc4
@param ImageFile_TotalCount
@desc Image source of Combo total damage digits (default: "cc5")
@type file
@dir img/SoRBatHUD/
@default cc5

@param ImageFile_RateText
@desc Image source of Combo "Rate" text (default: "cc6")
@type file
@dir img/SoRBatHUD/
@default cc6
@param ImageFile_RateCount
@desc Image source of Combo damage rate digits (default: "cc7")
@type file
@dir img/SoRBatHUD/
@default cc7

@param -Duration-
@param Combo_Duration
@desc Duration of consecutive hits are successful (default: 120)
@default 120
@type number
@param Combo_DurationTiming
@desc Timing to process the combo duration (default: 3)
@type select
@option Always (for fully real-time battle)
@value 0
@option Except for the command inputting
@value 1
@option Except for the action performing
@value 2
@option Except for commands and actions
@value 3
@default 3

@param -Combo Terminate-
@param StopCombo_MissAttack
@desc If true, consecutive hits are terminated when the attack is MISS.
@default false
@type boolean
@param StopCombo_TargetAction
@desc If true, consecutive hits are terminated when the target starts an action.
@default false
@type boolean
@param SuspendCombo_TargetGuard
@desc If true, consecutive hits are not counted while the target guards.
@default false
@type boolean

@param -Combo Damage Bonus-
@param ComboBonus_StartHit
@desc Initial hits that damage bonus is applied. Set less than 1 to disable. (default: 1)
@default 1
@type number
@param ComboBonus_DamageRate
@desc Damage bonus rate by combo(%) a in y=an+100. (default: 3)
@default 3
@type number

@param --Achievement--
@param Rewards_MaxHit
@desc Hits to activate a particular process. Set 0 to disable (default: 0)
@default 0
@type number
@param Rewards_Process
@desc A process (script text) activated when designated hits is achieved.
@default 
@type string

@param -Combo EXP Bonus-
@param UseComboBounsEXP
@desc If true, EXP bonus is applied according to the maximum consecutive hits in the battle.
@default false
@type boolean
@param Formula_ComboBounsEXP
@desc Formula of EXP Bonus (Polynominal of maximum combo hits "n")
@default n*(n+6)/6
@type string
*/

var SoR = SoR || {};

(function() {
const Param = PluginManager.parameters('SoR_BattleComboCounter_MZ');

const Combo_Ypadd = Number(Param['Combo_Y-padd'] || 0);
const Show_TotalDamage = Boolean(Param['Show_TotalDamage'] === 'true') || false;
const Show_ComboBonusRate = Boolean(Param['Show_ComboBonusRate'] === 'true') || false;


const ImageFile_Base = String(Param['ImageFile_Base']) || '';
const ImageFile_Count = String(Param['ImageFile_Count']) || '';
const ImageFile_Hits = String(Param['ImageFile_Hits']) || '';
const ImageFile_Total = String(Param['ImageFile_Total']) || '';
const ImageFile_TotalCount = String(Param['ImageFile_TotalCount']) || '';
const ImageFile_RateText = String(Param['ImageFile_RateText']) || '';
const ImageFile_RateCount = String(Param['ImageFile_RateCount']) || '';


const Rewards_MaxHit = Number(Param['Rewards_MaxHit'] || 0);
const Rewards_Process = String(Param['Rewards_Process'] || "");

const Combo_Duration = Number(Param['Combo_Duration'] || 0);
const Combo_DurationTiming = Number(Param['Combo_DurationTiming'] || 0);

const StopCombo_MissAttack = Boolean(Param['StopCombo_MissAttack'] === 'true') || false;
const StopCombo_TargetAction = Boolean(Param['StopCombo_TargetAction'] === 'true') || false;
const SuspendCombo_TargetGuard = Boolean(Param['SuspendCombo_TargetGuard'] === 'true') || false;

const ComboBonus_StartHit = Number(Param['ComboBonus_StartHit'] || 0);
const ComboBonus_DamageRate = Number(Param['ComboBonus_DamageRate'] || 0);

const UseComboBounsEXP = Boolean(Param['UseComboBounsEXP'] === 'true') || false;
const Formula_ComboBounsEXP = String(Param['Formula_ComboBounsEXP']) || '';

//1.01
const Total_Ypadd = Number(Param['Total_Y-padd'] || 0);
const Bonus_Ypadd = Number(Param['Bonus_Y-padd'] || 0);

ImageManager.loadBCCMaterial = function(filename) {
    return this.loadBitmap('img/SoRBatHUD/', filename, 0, true);
}
////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////
// Use them from other scripts
////////////////////////////////////////////////////////////////////////
Game_Battler.prototype.nHits = function() { // target current combo hits
	if(!$gameParty.inBattle()) return 0;
	return this.combo_data ? this.combo_data.Hitcount : 0;
}
Game_Battler.prototype.nHitsClear = function() { // clear target combo hits
	if(!$gameParty.inBattle()) return;
	BattleManager.forceClear_ComboCoutner(this);
}
BattleManager.nHits = function() { // current highest combo hits (displayed on the screen)
	if(!$gameParty.inBattle()) return 0;
	const tar = SceneManager._scene._SoR_ComboCounters.CountersSpr.battler;
	if(tar==null) return 0;
	return tar.combo_data ? tar.combo_data.Hitcount : 0;
}
BattleManager.GetMaxHits = function() {
	return this._rewards.maxhit;
}
Game_Party.prototype.GetMaxHits = function() {
	return this._maxComboHits;
}
////////////////////////////////////////////////////////////////////////
// End
////////////////////////////////////////////////////////////////////////







////////////////////////////////////////////////////////////////////////
const SoR_CC_BM_initMembers = BattleManager.initMembers;
BattleManager.initMembers = function() {
	SoR_CC_BM_initMembers.call(this);
	this._maxComboC = 0;
}

const SoR_CC_BM_makeRewards = BattleManager.makeRewards;
BattleManager.makeRewards = function() {
	SoR_CC_BM_makeRewards.call(this);
    this._rewards.maxhit = this._maxComboC;
}

const SoR_BCC_BM_FinalizeBattleRewards_midterm2 = BattleManager.FinalizeBattleRewards_midterm2;
BattleManager.FinalizeBattleRewards_midterm2 = function(){
	SoR_BCC_BM_FinalizeBattleRewards_midterm2.call(this);
	if(UseComboBounsEXP) this.calcEXPBonus_byComboCount();
}

BattleManager.calcEXPBonus_byComboCount = function(){
	const exptext = Formula_ComboBounsEXP.replace(new RegExp("n", 'g'), BattleManager._rewards.maxhit );
	this._rewards.ComboBonusExp = Math.floor(SoR_Eval(exptext));
	this._rewards.exp += this._rewards.ComboBonusExp;
}

const SoR_SL_GP_initialize = Game_Party.prototype.initialize;
Game_Party.prototype.initialize = function() {
    SoR_SL_GP_initialize.call(this);
    this._maxComboHits = 0;
}

const SoR_combo_GB_onBattleStart = Game_Battler.prototype.onBattleStart;
Game_Battler.prototype.onBattleStart = function(advantageous) {
	SoR_combo_GB_onBattleStart.call(this, ...arguments);
    this.combo_data = new SoR_ComboCounter();
}



////////////////////////////////////////////////////////////////////////
function SoR_ComboCounter() {
    this.initialize.apply(this, ...arguments);
}

SoR_ComboCounter.prototype.initialize = function() {
	this.visible = false;
	this.hitcountable = false;
	this.Hitcount = 0;
	this.TotalDamage = 0;
	this.DamageRatio = 100;
	this.duration = -1;
}

SoR_ComboCounter.prototype.resetCount = function() {
	this.initialize();
}

SoR_ComboCounter.prototype.RecordMaxHits = function() {
	if(!$gameParty._maxComboHits || this.Hitcount > $gameParty._maxComboHits){
		$gameParty._maxComboHits = this.Hitcount;
		
		if(Rewards_MaxHit) SoR_Eval(Rewards_Process);
	}
	if(this.Hitcount >= BattleManager._maxComboC) BattleManager._maxComboC = this.Hitcount; //current battle
	BattleManager.finalizeCurrentComboHits(this.Hitcount);
}
////////////////////////////////////////////////////////////////////////

BattleManager.finalizeCurrentComboHits = function(n){
	//for others
}


const SoR_BCC_GB_performActionStart = Game_Battler.prototype.performActionStart;
Game_Battler.prototype.performActionStart = function(action) {
    SoR_BCC_GB_performActionStart.call(this, ...arguments);
	if(StopCombo_TargetAction) BattleManager.forceClear_ComboCoutner(this);
}

const SoR_BCC_GA_apply = Game_Action.prototype.apply;
Game_Action.prototype.apply = function(target) {
	SoR_BCC_GA_apply.call(this, ...arguments);

	const result = target.result();
	if (result.missed || result.evaded) {
		if(StopCombo_MissAttack) BattleManager.forceClear_ComboCoutner(target);
	}
}

const SoR_combo_GA_executeDamage = Game_Action.prototype.executeDamage
Game_Action.prototype.executeDamage = function(target, value) {
	value = this.applyComboEffective(target, value);
	SoR_combo_GA_executeDamage.call(this,target, value);
}

Game_Action.prototype.applyComboEffective = function(target, value) {
		
	if($gameParty.inBattle()){
		const flag1 = (this.subject().isActor() && target.isEnemy());//
		const flag2 = (this.subject().isEnemy() && target.isActor());//
		if ((flag1 || flag2) && isGuarded_ComboContinue(target) ) {
			target.combo_data.DamageRatio = 100 + this.calcComboRate(target,value);
			value = parseInt( value *( target.combo_data.DamageRatio /100.0) );
			target.combo_data.hitcountable = true;
			target.combo_data.visible = true;
			target.combo_data.Hitcount += 1;
			target.combo_data.TotalDamage += value;
			target.combo_data.duration = Combo_Duration;
		}
	}

	return value;
}

Game_Action.prototype.calcComboRate = function(target, value) {
	if(ComboBonus_StartHit<=1) return 0;
	return (target.combo_data.Hitcount-ComboBonus_StartHit+1) * this.comboBonusBasis(target, value);
}
Game_Action.prototype.comboBonusBasis = function(target, value) {
	return ComboBonus_DamageRate;
}


function isGuarded_ComboContinue(target){
	if(!SuspendCombo_TargetGuard) return true;
	if(target.isGuard()) return false;
	return true;
}

BattleManager.update_ComboCoutner = function() {
	this.allBattleMembers().forEach(function(battler) {
	     this.update_BattlerComboCoutner(battler);	
    }, this);
}

BattleManager.update_BattlerComboCoutner = function(battler) {
	const cmb = battler.combo_data;
    	
    if (battler.isDead() && cmb.duration != -1) cmb.duration = 0; //terminate when enemy is dead
	
	if(isEnableCountComboDuration() && cmb.duration > 0) cmb.duration--;	
	if(cmb.duration == 0){
		if(battler.isEnemy()) cmb.RecordMaxHits();
		cmb.resetCount();
	}
}

function isEnableCountComboDuration(){
	switch(Combo_DurationTiming){
        case 1:
            if(BattleManager._inputting) return false;
        break;
        case 2:
            if(this._phase == "action") return false;
        break;
        case 3:
            if(this._phase == "action" || BattleManager._inputting) return false;
        break;
        default:
		return true;
        break;
    }
	
	return true;
}

BattleManager.forceClear_ComboCoutner = function(battler) {
	const cmb = battler.combo_data;
	cmb.duration = 0;
	if(battler.isEnemy()) cmb.RecordMaxHits();
	cmb.resetCount();
}

//////////////////////////////////////////////////////////////////////////////
const SoR_BCC_SB_createSpriteset = Scene_Battle.prototype.createSpriteset;
Scene_Battle.prototype.createSpriteset = function() {
	SoR_BCC_SB_createSpriteset.call(this);
    this.create_SoRComboCounters();
}

Scene_Battle.prototype.create_SoRComboCounters = function() {
    this._SoR_ComboCounters = new SoR_BattleComboCoutners();
	this.addChild(this._SoR_ComboCounters);
}

Scene_Battle.prototype.update_combosprites = function() {
	const bm = $gameTroop.members(); //$gameParty.battleMembers();
	let target = null;
	let max = 0;
	for(const btlr of bm){
		if(typeof btlr.combo_data === "undefined") continue; //for troop wave reinforcement
		const cmb = btlr.combo_data.visible;
		if(cmb && btlr.combo_data.Hitcount > max){
			max = btlr.combo_data.Hitcount;
			target = btlr;
		}
	}
	this._SoR_ComboCounters.changeTarget(target);
	this._SoR_ComboCounters.updatecounter();
}



//////////////////////////////////////////////////////////////////////////////
function SoR_BattleComboCoutners() {
    this.initialize.apply(this, arguments);
}
SoR_BattleComboCoutners.prototype = Object.create(Sprite.prototype);
SoR_BattleComboCoutners.prototype.constructor = SoR_BattleComboCoutners;

SoR_BattleComboCoutners.prototype.initialize = function(){
     Sprite.prototype.initialize.call(this);
     this.CreateCounters();
}


SoR_BattleComboCoutners.prototype.CreateCounters = function(){
	ImageManager.loadBCCMaterial(ImageFile_Base);
	ImageManager.loadBCCMaterial(ImageFile_Count);
	ImageManager.loadBCCMaterial(ImageFile_Hits);
	ImageManager.loadBCCMaterial(ImageFile_Total);
	ImageManager.loadBCCMaterial(ImageFile_TotalCount);
	ImageManager.loadBCCMaterial(ImageFile_RateText);
	ImageManager.loadBCCMaterial(ImageFile_RateCount);
	
	this.CountersSpr = new Sprite();
	this.CountersSpr.battler = null;
	this.CountersSpr.visible = true;
	this.addChild(this.CountersSpr);
	this.CountersSpr.cmb = new SoR_ComboCounterSprites();
	
	this.CountersSpr.addChild(this.CountersSpr.cmb.combo_sprite_bg);
	this.CountersSpr.addChild(this.CountersSpr.cmb.combo_sprite_base);
	this.CountersSpr.addChild(this.CountersSpr.cmb.combo_sprite_hittxt);
	this.CountersSpr.addChild(this.CountersSpr.cmb.combo_sprite_total);
	this.CountersSpr.addChild(this.CountersSpr.cmb.combo_sprite_ratio);
	this.CountersSpr.x = 0;
	this.CountersSpr.y = 0; 
	this.x = Graphics.boxWidth - 256;
	this.y = 40+Combo_Ypadd;
}


SoR_BattleComboCoutners.prototype.changeTarget = function(target) {
	if(!target) return;
	if(this.CountersSpr.battler != target) this.CountersSpr.battler = target;
}


SoR_BattleComboCoutners.prototype.updatecounter = function() {
	const battler = this.CountersSpr.battler;
	if(battler == null) return;
	
	const cmbsprs = this.CountersSpr.cmb;
	const cmbdata = battler.combo_data;
	
	if(cmbdata.visible == true){
		if(cmbdata.Hitcount>=2){
			cmbsprs.combo_sprite_base.opacity = 255;
			cmbsprs.combo_sprite_total.opacity = cmbsprs.combo_sprite_base.opacity;
			cmbsprs.combo_sprite_hittxt.opacity = cmbsprs.combo_sprite_base.opacity;
			if(cmbdata.DamageRatio > 100) cmbsprs.combo_sprite_ratio.opacity = cmbsprs.combo_sprite_base.opacity;
			
			cmbsprs.moveperformX = cmbsprs.moveperformX>0? cmbsprs.moveperformX/2 : 0;
			cmbsprs.combo_sprite_hittxt.x = cmbsprs.hittxtbaseX + cmbsprs.moveperformX;
			cmbsprs.combo_sprite_bg.scale.x = 1;
			cmbsprs.combo_sprite_bg.scale.y = 1;
		}
	}
	else if(cmbsprs.combo_sprite_base.opacity > 0){
		cmbsprs.combo_sprite_base.opacity -= 35;
		cmbsprs.combo_sprite_total.opacity = cmbsprs.combo_sprite_base.opacity;
		cmbsprs.combo_sprite_hittxt.opacity = cmbsprs.combo_sprite_base.opacity;
		if(cmbsprs.combo_sprite_ratio.opacity > cmbsprs.combo_sprite_base.opacity) cmbsprs.combo_sprite_ratio.opacity = cmbsprs.combo_sprite_base.opacity;
		
		cmbsprs.moveperformX = cmbsprs.moveperformX<1 ? 2 : cmbsprs.moveperformX*2;
		cmbsprs.combo_sprite_hittxt.x = cmbsprs.hittxtbaseX + cmbsprs.moveperformX;
	}
	else{
		cmbsprs.combo_sprite_base.opacity = 0;
		cmbsprs.combo_sprite_total.opacity = cmbsprs.combo_sprite_base.opacity;
		cmbsprs.combo_sprite_hittxt.opacity = cmbsprs.combo_sprite_base.opacity;
		cmbsprs.combo_sprite_ratio.opacity = cmbsprs.combo_sprite_base.opacity;
		
		for (let i = 0; i < cmbsprs.combo_sprite_hits.length; i++) cmbsprs.combo_sprite_hits[i].opacity = cmbsprs.combo_sprite_base.opacity;
	    for (let i = 0; i < cmbsprs.combo_sprite_totals.length; i++) cmbsprs.combo_sprite_totals[i].opacity = cmbsprs.combo_sprite_base.opacity;
		const oldhitnum = cmbsprs.combo_sprite_hits.length;
	    for(let j = 0; j < oldhitnum; j++) cmbsprs.combo_sprite_base.removeChild(cmbsprs.combo_sprite_hits[j]);
		const oldhittotal = cmbsprs.combo_sprite_totals.length;
		for(let j = 0; j < oldhittotal; j++) cmbsprs.combo_sprite_total.removeChild(cmbsprs.combo_sprite_totals[j]);
		const oldhitratio = cmbsprs.combo_sprite_ratios.length;
		for(let j = 0; j < oldhitratio; j++) cmbsprs.combo_sprite_ratio.removeChild(cmbsprs.combo_sprite_ratios[j]);
		cmbsprs.moveperformX = 128;
		cmbsprs.combo_sprite_bg.opacity = 0;
		return;
	}

    if(cmbdata.hitcountable == true && cmbdata.Hitcount>=2){
		const oldhitnum = cmbsprs.combo_sprite_hits.length;
	    for(let j = 0; j < oldhitnum; j++) cmbsprs.combo_sprite_base.removeChild(cmbsprs.combo_sprite_hits[j]);
		const oldhittotal = cmbsprs.combo_sprite_totals.length;
		for(let j = 0; j < oldhittotal; j++) cmbsprs.combo_sprite_total.removeChild(cmbsprs.combo_sprite_totals[j]);
		const oldhitratio = cmbsprs.combo_sprite_ratios.length;
		for(let j = 0; j < oldhitratio; j++) cmbsprs.combo_sprite_ratio.removeChild(cmbsprs.combo_sprite_ratios[j]);
		
	    cmbsprs.refresh_combonum_sprite(cmbdata);
		
		const newhitnum = cmbsprs.combo_sprite_hits.length;
		for(let j = 0; j < newhitnum; j++) cmbsprs.combo_sprite_base.addChild(cmbsprs.combo_sprite_hits[j]);
		const newhittotal = cmbsprs.combo_sprite_totals.length;
		for(let j = 0; j < newhittotal; j++) cmbsprs.combo_sprite_total.addChild(cmbsprs.combo_sprite_totals[j]);
		const newhitratio = cmbsprs.combo_sprite_ratios.length;
		for(let j = 0; j < newhitratio; j++) cmbsprs.combo_sprite_ratio.addChild(cmbsprs.combo_sprite_ratios[j]);
		
		cmbsprs.combo_sprite_bg.opacity = 255;
	}

	if(battler.combo_data.visible == true && cmbsprs.combo_sprite_base.opacity > 0){
	
			const nhits = cmbsprs.combo_sprite_hits.length;
			for (let i = 0; i < nhits; i++) {
			   if (cmbsprs.combo_sprite_hits[i].scale.x > 1.01) {
				   cmbsprs.combo_sprite_hits[i].scale.x -= 0.5;									
				   cmbsprs.combo_sprite_hits[i].scale.y = cmbsprs.combo_sprite_hits[i].scale.x
			   }
			   else{
				   cmbsprs.combo_sprite_hits[i].scale.x = 1.0;
				   cmbsprs.combo_sprite_hits[i].scale.y = 1.0;
			   }
			}

			const nratio = cmbsprs.combo_sprite_ratios.length;
			for (let i = 0; i < nratio; i++) {
			   if (cmbsprs.combo_sprite_ratios[i].scale.x > 1.01) {
				   cmbsprs.combo_sprite_ratios[i].scale.x -= 0.5;									
				   cmbsprs.combo_sprite_ratios[i].scale.y = cmbsprs.combo_sprite_ratios[i].scale.x
			   }
			   else{
				   cmbsprs.combo_sprite_ratios[i].scale.x = 1.0;
				   cmbsprs.combo_sprite_ratios[i].scale.y = 1.0;
			   }
			}
			
			if(cmbsprs.combo_sprite_bg.opacity > 0.1){
				cmbsprs.combo_sprite_bg.opacity /= 1.2;
				cmbsprs.combo_sprite_bg.scale.x *= 1.28;
				cmbsprs.combo_sprite_bg.scale.y *= 1.22;
			}		
	}
}

////////////////////////////////////////////////////////////////////////
function SoR_ComboCounterSprites(){
	this.initialize.apply(this, arguments);
}


SoR_ComboCounterSprites.prototype.initialize = function() {
		this.comboInteger = 0;
     	this.combo_sprite_base = new Sprite(ImageManager.loadBCCMaterial(ImageFile_Base));
		this.combo_sprite_bg = new Sprite(ImageManager.loadBCCMaterial(ImageFile_Base));

		this.combo_sprite_hits = [];
		this.combo_sprite_hitnum = ImageManager.loadBCCMaterial(ImageFile_Count); 
		this.combo_sprite_hittxt = new Sprite(ImageManager.loadBCCMaterial(ImageFile_Hits));

		this.combo_sprite_totals = [];
		if(Show_TotalDamage){
				this.combo_sprite_total = new Sprite(ImageManager.loadBCCMaterial(ImageFile_Total));
				this.combo_sprite_totalnum = ImageManager.loadBCCMaterial(ImageFile_TotalCount);
		}
		else{
			this.combo_sprite_total = new Sprite();
			this.combo_sprite_totalnum = new Bitmap(1,1);
		}
		
		this.combo_sprite_ratios = [];
		if(Show_ComboBonusRate){
				this.combo_sprite_ratio = new Sprite(ImageManager.loadBCCMaterial(ImageFile_RateText));
				this.combo_sprite_rationum = ImageManager.loadBCCMaterial(ImageFile_RateCount);
		}
		else{
			this.combo_sprite_ratio = new Sprite();
			this.combo_sprite_rationum = new Bitmap(1,1);
		}
				
		
		this.hittxtbaseX = 190;
		this.moveperformX = 128;
		this.combo_sprite_hittxt.x = this.hittxtbaseX + this.moveperformX;
		this.combo_sprite_hittxt.y = 40;
		
		this.combo_sprite_base.opacity = 0;
		this.combo_sprite_total.opacity = 0;
		this.combo_sprite_hittxt.opacity = 0;
		this.combo_sprite_ratio.opacity = 0;
		this.combo_sprite_bg.opacity = 0;
		this.combo_sprite_bg.anchor.x = 0.5;
		this.combo_sprite_bg.anchor.y = 0.5;
		this.combo_sprite_bg.x = 128;
		this.combo_sprite_bg.y = 36;
		
		this.combo_sprite_total.x = 0;
		this.combo_sprite_total.y = 68 + Total_Ypadd;
		this.combo_sprite_ratio.x = 0;
		this.combo_sprite_ratio.y = 90 + Total_Ypadd + Bonus_Ypadd;
}

SoR_ComboCounterSprites.prototype.refresh_combonum_sprite = function(cmbdata) {
	let Numimg = this.combo_sprite_hitnum;
	let wd = Numimg.width / 10;
	let ht = Numimg.height;
	let numarr =  Math.abs(cmbdata.Hitcount).toString().split("");
	this.comboInteger = cmbdata.Hitcount;
	
	this.combo_sprite_hits.length = 0;
    for (let i = 0; i <  numarr.length; i++) {
		const ndigit = Number(numarr[i]);
		     this.combo_sprite_hits[i] = new Sprite(Numimg);
			 this.combo_sprite_hits[i].setFrame(ndigit * wd, 0, wd, ht);
		     this.combo_sprite_hits[i].x = (i * (wd-10)) - Math.floor(((numarr.length-1) * (wd)))/2 +132; //-12;
		     this.combo_sprite_hits[i].y = 36;
			 this.combo_sprite_hits[i].anchor.x = 0.5;
			 this.combo_sprite_hits[i].anchor.y = 0.5;
		     this.combo_sprite_hits[i].scale.x = 4.0;
			 this.combo_sprite_hits[i].scale.y = 4.0;
			 this.combo_sprite_hits[i].opacity = this.combo_sprite_base.opacity; 
			 this.combo_sprite_hits[i].wd = wd; //width one digit
	}
	
	//this.combo_sprite_totalnum
	Numimg = this.combo_sprite_totalnum;
	wd = Numimg.width / 10;
	ht = Numimg.height;
	numarr =  Math.abs(cmbdata.TotalDamage).toString().split("");
	this.combo_sprite_totals.length = 0;
	
    for (let i = 0; i <  numarr.length; i++) {
		const ndigit = Number(numarr[i]);
		     this.combo_sprite_totals[i] = new Sprite(Numimg);
			 this.combo_sprite_totals[i].setFrame(ndigit * wd, 0, wd, ht);
			 this.combo_sprite_totals[i].x = (i * (wd-4)) - Math.floor(((numarr.length-1) * (wd))) + 256; 
		     this.combo_sprite_totals[i].y = 13;
			 this.combo_sprite_totals[i].anchor.x = 0.5;
			 this.combo_sprite_totals[i].anchor.y = 0.5;
		     this.combo_sprite_totals[i].scale.x = 1.0;
			 this.combo_sprite_totals[i].scale.y = 1.0;
			 this.combo_sprite_totals[i].opacity = this.combo_sprite_base.opacity;			 
			 this.combo_sprite_totals[i].wd = wd; //width one digit
	}
	
	
	//this.combo_sprite_rationum
	Numimg = this.combo_sprite_rationum;
	wd = Numimg.width / 10;
	ht = Numimg.height;
	numarr =  Math.abs(cmbdata.DamageRatio).toString().split("");
	this.combo_sprite_ratios.length = 0;
	
    for (let i = 0; i <  numarr.length; i++) {
		const ndigit = Number(numarr[i]);
		     this.combo_sprite_ratios[i] = new Sprite(Numimg);
			 this.combo_sprite_ratios[i].setFrame(ndigit * wd, 0, wd, ht);
			 this.combo_sprite_ratios[i].x = (i * (wd-5)) - Math.floor(((numarr.length-1) * (wd))) + 232; 
		     this.combo_sprite_ratios[i].y = 13;
			 this.combo_sprite_ratios[i].anchor.x = 0.5;
			 this.combo_sprite_ratios[i].anchor.y = 0.5;
		     this.combo_sprite_ratios[i].scale.x = 3.0;
			 this.combo_sprite_ratios[i].scale.y = 3.0;
			 this.combo_sprite_ratios[i].opacity = this.combo_sprite_base.opacity;			 
			 this.combo_sprite_ratios[i].wd = wd; //width one digit
	}
	
	cmbdata.hitcountable = false;
}
////////////////////////////////////////////////////////////////////////




const SoR_SB_updateBattleProcess = Scene_Battle.prototype.updateBattleProcess;
Scene_Battle.prototype.updateBattleProcess = function() {
	SoR_SB_updateBattleProcess.call(this);
   
    this.update_combosprites();
    BattleManager.update_ComboCoutner();
}


////////////////////////////////////////////////////////////////////////
function SoR_Eval(ev) {
    const sentence = "return (" + ev + ");";

    if(typeof $gameTemp.SoRTmp_script === "undefined") $gameTemp.SoRTmp_script = new Map();
    if(!$gameTemp.SoRTmp_script.has(sentence)){
        $gameTemp.SoRTmp_script.set(sentence, new Function(sentence));
    }     
    const res = $gameTemp.SoRTmp_script.get(sentence)();
    return res;
}


}());