//=============================================================================
// SoR_BattleActionParameterRefactor_MZ.js
// SoR License (C) 2020 蒼竜, REQUIRED User Registration on Dragon Cave
// http://dragonflare.blue/dcave/license.php
// ----------------------------------------------------------------------------
// Latest version v1.11 (2021/10/13)
//=============================================================================
/*:ja
@plugindesc ＜戦闘行動パラメータ参照細分化＞ v1.11
@author 蒼竜
@target MZ
@url https://dragonflare.blue/dcave/
@help 戦闘中のスキルなどの行動実行時のパラメータ参照処理を
細分化します。RPGツクールのデフォルトシステムでは仕様上、
プラグインによる機能拡張性に乏しく、プラグイン単位での競合のリスク管理が
煩雑であるため、根本の修正を行います。

"ゲームそのものには何の変化もありません"が、
他のスキル設定に干渉する拡張プラグインのための基幹システムとなります。
できるだけプラグインマネージャ上部へ入れてください。
(戦闘システムはコアスクリプトの実装依存が強いため、
他者作戦闘システム関連プラグインとの共用時に
細分化による機能を取り入れるためにコード置き換えが必要な場合があります。
実用上の問題発生など、詳しくはサポートまで問い合わせください。)

※今後の「スキル設定」に干渉するプラグインたちの基盤となる
必須の前提プラグインとなります。

※関連プラグインの状況に応じて、拡張・更新があります。
関連プラグインの導入・更新においては、本プラグインも常に
「最新の状態」である必要があります。

@param Overwrite_MPTPcostDef
@desc 'true'でMP/TP消費を再定義する (default: true)
@default true
@type boolean
@param Overwrite_ActionSpeedDef
@desc 'true'でスキルの"速度"設定に関する基本処理を再定義する (default: true)
@default true
@type boolean
@param Overwrite_ActionBase
@desc 'true'でスキル行動の基本処理そのものを再定義・拡張する (default: true)
@default true
@type boolean
@param Overwrite_BMEndAction
@desc 'true'で戦闘アクションの終了時処理そのものを再定義・拡張する (default: true)
@default true
@type boolean
@param Overwrite_ExecuteDamage
@desc 'true'でダメージ処理の発生時処理そのものを再定義・拡張する (default: true)
@default true
@type boolean
*/
/*:
@plugindesc <Battle Action Parameter Refactor> v1.11
@author Soryu
@target MZ
@url https://dragonflare.blue/dcave/index_e.php
@help This plugin modifies the default implementation in battle actions
to refer related parameters and make system extensions.
by classifying each mechanism into more smaller chanks of process.

Modification of battle core mechanism by each plugin may emerge
great risk due to poor extendability in default RPGMaker system.

Just this plugin actually gives "NO effect" to the games,
which is a fundamental treatment for other plugins to extend 
the battle defeated process.

!! This is a prerequisite (base) plugin for upcoming plugins which approach
to the battle action mechanism.

!! This plugin will be updated and extended according to the release and 
update of other related plugin. Therefore, this must be always UP-TO-DATE.

!! For combination with plugins by other developers, special treatment
sometimes may be required since the original implementation in the 
battle system is well optimized for orthodx command battle RPG.
Those who have stacked due to raised issues in pracitcal usage 
may ask a help for the support.
@param Overwrite_MPTPcostDef
@desc If true, MP/TP consumption is re-defined for extension. (default: true)
@default true
@type boolean
@param Overwrite_ActionSpeedDef
@desc If true, a mechanism for skill speed is re-defined for extension. (default: true)
@default true
@type boolean
@param Overwrite_ActionBase
@desc If true, a fundamental mechanism for skill management is re-defined for extension. (default: true)
@default true
@type boolean
@param Overwrite_BMEndAction
@desc If true, a process handling battle action end is re-defined for extension. (default: true)
@default true
@type boolean
@param Overwrite_ExecuteDamage
@desc If true, a process of damage dealt is re-defined for extension. (default: true)
@default true
@type boolean
*/
(function() {
const Param = PluginManager.parameters('SoR_BattleActionParameterRefactor_MZ');
const Overwrite_MPTPcostDef = Boolean(Param['Overwrite_MPTPcostDef'] === 'true' || false);
const Overwrite_ActionSpeedDef = Boolean(Param['Overwrite_ActionSpeedDef'] === 'true' || false);
const Overwrite_ActionBase = Boolean(Param['Overwrite_ActionBase'] === 'true' || false);
const Overwrite_BMEndAction = Boolean(Param['Overwrite_BMEndAction'] === 'true' || false);
const Overwrite_ExecuteDamage = Boolean(Param['Overwrite_ExecuteDamage'] === 'true' || false);

////////////////////////////////////////////////////////////////////////////
// Default Skill Cost
////////////////////////////////////////////////////////////////////////////
// Interface
Game_BattlerBase.prototype.skillMpCostEx = function(skill) {
    return skill.mpCost;
}
Game_BattlerBase.prototype.correctSkillMPCost = function(base) {
    let finalized = Math.floor(base * this.mcr);
    return finalized;
}

Game_BattlerBase.prototype.skillTpCostEx = function(skill) {
    return skill.tpCost;
}
Game_BattlerBase.prototype.correctSkillTPCost = function(base) {
    let finalized = base;
    return finalized;
}


////////////////////////////////////////////////////////////////////////////
// Core
if(Overwrite_MPTPcostDef){
    Game_BattlerBase.prototype.skillMpCost = function(skill) {
        let ret = this.skillMpCostEx(skill);
        ret = this.correctSkillMPCost(ret);
        return ret;
    }

    Game_BattlerBase.prototype.skillTpCost = function(skill) {
        let ret = this.skillTpCostEx(skill);
        ret = this.correctSkillTPCost(ret);
        return ret;
    }
}

////////////////////////////////////////////////////////////////////////////
// Action Speed
////////////////////////////////////////////////////////////////////////////
// Interface
Game_Battler.prototype.GetActionSpeed = function(item) {// <-- basis
    return item.speed;
}
Game_Battler.prototype.CorrectTPBDelay = function(delay) {
    return Math.sqrt(delay)/ this.tpbSpeed();
}
Game_Action.prototype.MakeSpeedBase = function(agi){
    return agi + Math.randomInt(Math.floor(5 + agi / 4));
}

////////////////////////////////////////////////////////////////////////////
// Core
if(Overwrite_ActionSpeedDef){
    Game_Battler.prototype.tpbRequiredCastTime = function() {
        const actions = this._actions.filter(action => action.isValid());
        const items = actions.map(action => action.item());
        const delay = items.reduce((r, item) => r + Math.max(0, -this.GetActionSpeed(item)), 0);
        const time = this.CorrectTPBDelay(delay);
        return time;
    }

    Game_Action.prototype.speed = function() {
        const agi = this.subject().agi;
        let speed = this.MakeSpeedBase(agi);
        if (this.item()) speed += this.subject().GetActionSpeed(this.item());
        if (this.isAttack()) speed += this.subject().attackSpeed();
        return speed;
    }
}
////////////////////////////////////////////////////////////////////////////

if(Overwrite_ActionBase){
Game_Action.prototype.apply = function(target) {
    this.tmpTarget = target;///
    let result = this.apply_resetTarget(target);
    result.used = this.apply_checkUsed(target,result);
    result.missed = this.apply_checkMissed(target,result);
    result.evaded = this.apply_checkEvaded(target,result);
    result.physical = this.apply_checkAttackType(target,result);
    result.drain = this.apply_checkDrainAttack(target,result);
    result = this.apply_finalzeBeforeHit(target, result);

    if (result.isHit()) {
        if (this.item().damage.type > 0) {
            result.critical = this.apply_checkCritical(target,result);
            let value = this.makeDamageValue(target, result.critical);
            value = this.correctDamageValue(target, value, result.critical);
            this.executeDamage(target, value);
            this.executeFinalizeAfterDamage(target, value, result);
        }
        this.applyItemEffects(target);
    }
    else{
        this.executeProcessMiss(target, result);
    }
    this.updateLastTarget(target);
    this.FinalizeAfterApply(target, result);
}

Game_Action.prototype.apply_resetTarget = function(target) {
    const result = target.result();
    this.subject().clearResult();
    result.clear();
    return result;
}
Game_Action.prototype.apply_checkUsed = function(target,result) {
    return this.testApply(target);
}
Game_Action.prototype.apply_checkMissed = function(target,result) {
    return result.used && Math.random() >= this.itemHit(target);
}
Game_Action.prototype.apply_checkEvaded = function(target,result) {
    return !result.missed && Math.random() < this.itemEva(target);
}
Game_Action.prototype.apply_checkAttackType = function(target,result) {
    return this.isPhysical();
}
Game_Action.prototype.apply_checkDrainAttack = function(target,result) {
    return this.isDrain();
}

Game_Action.prototype.apply_finalzeBeforeHit = function(target,result) {
    /*For other plugins extension*/
    return result;
}

Game_Action.prototype.apply_checkCritical = function(target,result) {
    return Math.random() < this.itemCri(target);
}
Game_Action.prototype.correctDamageValue = function(target, value, IsCri) {
    /*For other plugins extension*/
    return value;
}
Game_Action.prototype.executeFinalizeAfterDamage = function(target, value,result) {
    /*For other plugins extension*/
}
Game_Action.prototype.applyItemEffects = function(target) {
    for (const effect of this.item().effects) {
        this.applyItemEffect(target, effect);
    }
    this.applyItemUserEffect(target);
}


Game_Action.prototype.executeProcessMiss = function(target, result) {
    /*For other plugins extension*/
}


Game_Action.prototype.FinalizeAfterApply = function(target, result) {
    /*For other plugins extension*/
}
}


////////////////////////////////////////////////////////////////////////////
if(Overwrite_BMEndAction){
BattleManager.endAction = function() {
    this.endAction_extended();
    this._logWindow.endAction(this._subject);
    this._phase = "turn";
    if (this._subject.numActions() === 0) {
        this.endBattlerActions(this._subject);
        this._subject = null;
    }
}
}
BattleManager.endAction_extended = function() {/*others*/}


if(Overwrite_ExecuteDamage){

Game_Action.prototype.executeHpDamage = function(target, value) {
    if (this.isDrain()) value = this.executeHpDrain(...arguments);
    this.makeSuccess(target);
    this.HpDamageDealtCore(...arguments);
    if (value > 0) this.TargetActuallHPDamageDealt(...arguments);
    this.executeDrainedHPprocess(...arguments);
}

Game_Action.prototype.executeHpDrain = function(target, value) {
    return value = Math.min(target.hp, value);
}

Game_Action.prototype.TargetActuallHPDamageDealt = function(target, value) {
    target.onDamage(value);
}

Game_Action.prototype.HpDamageDealtCore = function(target, value) {
    target.gainHp(-value);
}

Game_Action.prototype.executeDrainedHPprocess = function(target, value) {
    this.gainDrainedHp(value);
}

}



})();